#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

DKIM_VERIFY_PID="/var/run/mailplus_server/opendkim_verify.pid"
DKIM_VERIFY_CONF="${MAIL_SERVER_PKG}/target/etc/opendkim_verify.conf"
DKIM_VERIFY_BIN="${MAIL_SERVER_PKG}/target/sbin/opendkim -x ${DKIM_VERIFY_CONF}"
DKIM_VERIFY_LOCK="/var/lock/mailplus_server/opendkim_verify.lock"
DKIM_VERIFY_SOCK="/var/spool/@MailPlus-Server/opendkim_verify.sock"

checkStarted()
{
	if [ -f ${DKIM_VERIFY_PID} ]; then
		PID=`cat ${DKIM_VERIFY_PID}`
		PROCESS=`ps --pid ${PID} -o pid,args | grep opendkim | grep verify`
	fi

	if [ ! -z "${PROCESS}" ]; then
		return 1
	fi
	return 0
}

bin_status()
{
	if [ -f "${DKIM_VERIFY_PID}" ]; then
		PID=`cat ${DKIM_VERIFY_PID}`
		PROCESS=`cat "/proc/${PID}/cmdline" | grep "opendkim" | grep "verify"`

		if [ ! -z "${PROCESS}" ]; then
			return ${SERVICE_RUNNING}
		fi
		return ${SERVICE_DEAD_FPID}
	else
		if [ 0 != `ps -C opendkim -o pid,args | grep "${DKIM_VERIFY_CONF}" | wc -l` ] ; then
			return ${SERVICE_RUNNING}
		fi
		return ${SERVICE_NOT_RUNNING}
	fi > /dev/null
}

conf_status()
{
	checkConfKey "enable_dkim"
	return $?
}

status()
{
	conf_status
	local opendkim_enable=$?
	bin_status
	local opendkim_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${opendkim_status} ] && [ "${RUNKEY_DISABLE}" -ne "${opendkim_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${opendkim_status} ] && [ "${RUNKEY_ENABLE}" -ne "${opendkim_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	return ${opendkim_status}
}

runDaemon()
{
	rm -f "${DKIM_VERIFY_SOCK}"
	su MailPlus-Server -s /bin/sh -c "${DKIM_VERIFY_BIN}"

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status opendkim (verify): ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			return ${START_SUCCESSFUL}
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return ${START_FAILED}
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return ${START_FAILED}
}

start()
{
	(
	if flock -x 8; then
		checkStarted
		if [ 1 -eq $? ] ; then
			echo "opendkim (verify) is already running"
		else
			runDaemon
		fi

		flock -u 8
		rm ${DKIM_VERIFY_LOCK}
	fi
	)8> ${DKIM_VERIFY_LOCK}
}

checkStopped()
{
	local stopped="no"
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		if [ 0 == `ps -C opendkim -o pid,args | grep "${DKIM_VERIFY_CONF}" | wc -l` ] ; then
			# opendkim (verify) is terminated
			stopped="yes"
			break
		fi
		sleep "${SLEEP_TIME}"
	done

	if [ "yes" != "${stopped}" ]; then
		echo "Force kill opendkim (verify)"
		ps -C opendkim -o pid,args | grep "${DKIM_VERIFY_CONF}" | awk '{kill -9 $1}'
	fi
}

stopDaemon()
{
	kill `cat ${DKIM_VERIFY_PID}` &> /dev/null
	checkStopped
}

stop()
{
	(
	if flock -x 8; then
		stopDaemon

		flock -u 8
		rm ${DKIM_VERIFY_LOCK}
	fi
	)8> ${DKIM_VERIFY_LOCK}
}

restart()
{
	(
	if flock -x 8; then
		stopDaemon
		sleep 1
		runDaemon

		flock -u 8
		rm ${DKIM_VERIFY_LOCK}
	fi
	)8> ${DKIM_VERIFY_LOCK}
}

reload()
{
	ps -C opendkim -o pid,args | grep "${DKIM_VERIFY_CONF}" | awk '{kill -SIGUSR1 $1}'
}

composePID()
{
	local pid=$(cat ${DKIM_VERIFY_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/opendkim_verify.sh"
}


case "$1" in
	start)
		startOperation "mailserver_service_opendkim_verify"
		;;
	restart)
		restartOperation "mailserver_service_opendkim_verify"
		;;
	stop)
		stopOperation "mailserver_service_opendkim_verify"
		;;
	reload)
		reloadOperation "mailserver_service_opendkim_verify"
		;;
	status)
		status
		;;
	*)
		echo "Usage: $0 {start|stop|restart|status}" >&2
		exit 1
		;;
esac
