#!/bin/sh

. /var/packages/MailPlus-Server/target/backend_hook/hookUtils.conf
. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

CACHE_DIR="${MAIL_SERVER_PKG}/etc"

ANTI_VIRUS_ENGINE_CLAMAV="clamav"
ANTI_VIRUS_ENGINE_MCAFEE="synoav-mcafee"

SYNOAV_MCAFEE_SCRIPT="${MAIL_SERVER_PKG}/target/scripts/daemon/synoav_mcafee.sh"

MCAFEE_UPDATED="updated"
MCAFEE_OUTDATED="outdated"

CLAMAV_URL="current.cvd.clamav.net"
CLAMAV_DB_CLD="${MAIL_SERVER_PKG}/target/share/clamav/daily.cld"
CLAMAV_DB_CVD="${MAIL_SERVER_PKG}/target/share/clamav/daily.cvd"
CLAMAV_SIGTOOL="${MAIL_SERVER_PKG}/target/bin/sigtool"

# db path may be changed when spamassassin upgraded
SA_DB="${MAIL_SERVER_PKG}/target/var/spamassassin/3.004001/updates_spamassassin_org.cf"
SA_UPDATE="${MAIL_SERVER_PKG}/target/bin/sa-update"

DNSBL_CACHE="${CACHE_DIR}/dnsbl_status.json"

STATUS_UPDATED=0
STATUS_DISABLED=1
STATUS_OUTDATED=2

BUFFER_TIME=86400

ANTI_VIRUS_STATUS_CACHE="${CACHE_DIR}/anti_virus_status.json"
ANTI_SPAM_STATUS_CACHE="${CACHE_DIR}/anti_spam_status.json"
DNSBL_STATUS_CACHE="${CACHE_DIR}/dnsbl_status.json"

check_anti_spam_db_updated()
{
	local last_update_time
	local now_time
	local code
	local result

	if [ ! -f "${SA_DB}" ]; then
		return ${STATUS_OUTDATED}
	fi

	last_update_time=`stat -c "%Y" "${SA_DB}"`
	now_time=`date "+%s"`

	if [ "${now_time}" -gt `expr ${last_update_time} + ${BUFFER_TIME}` ]; then
		# possibly outdated, need to do version check

		${SA_UPDATE} --checkonly
		code=$?
		if [ "${code}" -ne "1" ]; then
			return ${STATUS_OUTDATED}
		fi
	fi

	return ${STATUS_UPDATED}
}

check_anti_spam()
{
	local code
	local cache_tmp="${ANTI_SPAM_STATUS_CACHE}.tmp.$$"

	check_anti_spam_db_updated
	code=$?

	if [ "${code}" -eq "${STATUS_OUTDATED}" ]; then
		jq -n --argjson val '"outdated"' '{"db_status": $val}' > "${cache_tmp}"
	elif [ "${code}" -eq "${STATUS_UPDATED}" ]; then
		jq -n --argjson val '"updated"' '{"db_status": $val}' > "${cache_tmp}"
	fi

	chown MailPlus-Server:MailPlus-Server ${cache_tmp}
	mv "${cache_tmp}" "${ANTI_SPAM_STATUS_CACHE}"
}

check_anti_virus_db_updated()
{
	local last_update_time
	local new_ver
	local old_ver
	local now_time
	local db
	local anti_virus_engine="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "anti_virus_engine")"

	if [ "${ANTI_VIRUS_ENGINE_MCAFEE}" == "${anti_virus_engine}" ]; then
		local synoav_mcafee_status="$(${SYNOAV_MCAFEE_SCRIPT} "check_update")"

		if [ "${MCAFEE_UPDATED}" == "${synoav_mcafee_status}" ]; then
			return ${STATUS_UPDATED}
		else
			return ${STATUS_OUTDATED}
		fi
	elif [ "${ANTI_VIRUS_ENGINE_CLAMAV}" == "${anti_virus_engine}" ]; then
		if [ -f "${CLAMAV_DB_CLD}" ]; then
			db="${CLAMAV_DB_CLD}"
		fi

		if [ -f "${CLAMAV_DB_CVD}" ]; then
			db="${CLAMAV_DB_CVD}"
		fi

		if [ -z "${db}" ]; then
			return ${STATUS_OUTDATED}
		fi

		last_update_time=`stat -c "%Y" "${db}"`
		now_time=`date "+%s"`

		if [ "${now_time}" -gt `expr ${last_update_time} + ${BUFFER_TIME}` ]; then
			# possibly outdated, need to do version check

			new_ver=`nslookup -query=txt ${CLAMAV_URL} | grep text | awk 'BEGIN {FS=":"} {print $3}'`
			old_ver=`${CLAMAV_SIGTOOL} -i "${db}" | grep Version | awk 'BEGIN {FS=": "} {print $2}'`

			if [ "${new_ver}" -gt "${old_ver}" ]; then
				return ${STATUS_OUTDATED}
			fi
		fi

		return ${STATUS_UPDATED}
	fi
}

check_anti_virus()
{
	local code
	local cache_tmp="${ANTI_VIRUS_STATUS_CACHE}.tmp.$$"

	check_anti_virus_db_updated
	code=$?

	if [ "${code}" -eq "${STATUS_OUTDATED}" ]; then
		jq -n --argjson val '"outdated"' '{"db_status": $val}' > "${cache_tmp}"
	elif [ "${code}" -eq "${STATUS_UPDATED}" ]; then
		jq -n --argjson val '"updated"' '{"db_status": $val}' > "${cache_tmp}"
	fi

	chown MailPlus-Server:MailPlus-Server ${cache_tmp}
	mv "${cache_tmp}" "${ANTI_VIRUS_STATUS_CACHE}"
}

check_dnsbl_blacklisted()
{
	local tid=`synowebapi --exec api=SYNO.MailPlusServer.Diagnosis method=blacklist_check 2>/dev/null | jq -r '.data.task_id'`
	local result
	local TRY_LIMIT=60 # 10 mins
	local SLEEP_TIME=10
	local cache_tmp="${DNSBL_CACHE}.tmp.$$"

	for i in `seq 0 1 "${TRY_LIMIT}"`
	do
		synowebapi --exec api=SYNO.MailPlusServer.Diagnosis method=blacklist_check_status task_id=\"${tid}\" 2>/dev/null > "${cache_tmp}"
		result=`cat "${cache_tmp}" | jq -r '.success'`
		if [ "${result}" != "true" ]; then
			# something wrong
			break
		fi

		result=`cat "${cache_tmp}" | jq -r '.data.status'`
		if [ "${result}" = "checking" ]; then
			sleep "${SLEEP_TIME}"
		elif [ "${result}" = "finish" ]; then
			break
		fi
	done

	chown MailPlus-Server:MailPlus-Server ${cache_tmp}
	mv "${cache_tmp}" "${DNSBL_CACHE}"
}

case "$1" in
	anti_spam)
		check_anti_spam
		;;
	anti_virus)
		check_anti_virus
		;;
	dnsbl)
		check_dnsbl_blacklisted
		;;
	all)
		check_anti_spam
		check_anti_virus
		check_dnsbl_blacklisted
		;;
	*)
		echo "Usage: $0 {anti_spam|anti_virus|dnsbl|all}" >&2
		exit 1
		;;
esac
