package MailPlusServer::Log;

use strict;
use warnings;

use Encode;
use JSON;
use MailPlusServer::Util;

sub Configure {
	$MailPlusServer::Log::logger = shift;
}

sub ErrorLog {
	if (defined $MailPlusServer::Log::logger) {
		$MailPlusServer::Log::logger->('err', @_);
	}
}

sub InfoLog {
	if (defined $MailPlusServer::Log::logger) {
		$MailPlusServer::Log::logger->('info', @_);
	}
}

sub get_msg_id {
	my $entity = shift;
	my $msg_id = '';

	if (defined($entity->head->get('Message-ID'))) {
		$msg_id = $entity->head->get('Message-ID');
		chomp($msg_id);
	}

	return $msg_id;
}

sub gen_security_log_text {
	my $msg_href = shift;

	my $text = to_json($msg_href);
	$text =~ s/^{/{"id":"SYNOMailPlus-Server",/;

	return $text;
}

sub security_log {
	my ($msg_href, $recipients_aref) = @_;

	$msg_href->{sender} = MailPlusServer::Util::trim_angle_bracket($msg_href->{sender});
	$msg_href->{sender} = MailPlusServer::Util::convert_to_eai_addr($msg_href->{sender});
	$msg_href->{subject} = Encode::decode('MIME-Header', $msg_href->{subject});
	if (!Encode::is_utf8($msg_href->{subject})) {
		$msg_href->{subject} = Encode::decode('utf8', $msg_href->{subject});
	}

	foreach my $recipient (@{$recipients_aref}) {
		my $Recipient = MailPlusServer::Util::trim_angle_bracket($recipient);
		$msg_href->{recipient} = MailPlusServer::Util::convert_to_eai_addr($Recipient);

		InfoLog(gen_security_log_text($msg_href));
	}
}

sub VirusLog {
	my ($sender, $recipients_aref, $senderip, $subject, $entity) = @_;

	my %msg;
	my $msg_id = get_msg_id($entity);
	my $report = '';
	my $status = $entity->head->get($MailPlusServer::Util::HeaderVirusStatus);

	if (defined($status) && $status =~ /, (.*) *$/) {
		$report = $1;
	}

	$msg{type} = "virus";
	$msg{src} = $senderip;
	$msg{sender} = $sender;
	$msg{subject} = $subject;
	if ($msg_id ne '') {
		$msg{event} = "Found virus in $msg_id ($report)";
	} else {
		$msg{event} = "Found virus ($report)";
	}

	security_log(\%msg, $recipients_aref);
}

sub SpamLog {
	my ($sender, $recipients_aref, $senderip, $subject, $entity) = @_;

	my %msg;
	my $msg_id = get_msg_id($entity);
	my $report = '';
	my $status = $entity->head->get($MailPlusServer::Util::HeaderSpamStatus);

	if (defined($status)) {
		$report = $status;
		$report =~ s/\s+$//;
	}

	$msg{type} = "spam";
	$msg{src} = $senderip;
	$msg{sender} = $sender;
	$msg{subject} = $subject;
	if ($msg_id ne '') {
		$msg{event} = "Detected spam in $msg_id ($report)";
	} else {
		$msg{event} = "Detected spam ($report)";
	}

	security_log(\%msg, $recipients_aref);
}

sub MCPLog {
	my ($sender, $recipients_aref, $senderip, $subject, $entity, $rules_aref) = @_;

	my %msg;
	my $msg_id = get_msg_id($entity);
	my $report = '';
	my $status = $entity->head->get($MailPlusServer::Util::HeaderMCPStatus);

	if (defined($status) && $status =~ /, (.*)$/) {
		$report = $1;
	}

	my @mcp_rules = split(', ', $report);
	foreach my $mcp_rule (@mcp_rules) {
		if ($mcp_rule =~ /SYNO_MCP_(\d*)/ && $1 < scalar(@{$rules_aref})) {
			$mcp_rule = $rules_aref->[$1]->{name};
		}
	}
	$report = join(', ', @mcp_rules);

	$msg{type} = "mcp";
	$msg{src} = $senderip;
	$msg{sender} = $sender;
	$msg{subject} = $subject;
	if ($msg_id ne '') {
		$msg{event} = "MCP for $msg_id ($report)";
	} else {
		$msg{event} = "MCP ($report)";
	}

	security_log(\%msg, $recipients_aref);
}

sub DangerousContentLog {
	my ($sender, $recipients_aref, $senderip, $subject, $entity, $report) = @_;

	my %msg;
	my $msg_id = get_msg_id($entity);

	$msg{type} = "content";
	$msg{src} = $senderip;
	$msg{sender} = $sender;
	$msg{subject} = $subject;
	if ($msg_id ne '') {
		$msg{event} = "Content scan for $msg_id ($report)";
	} else {
		$msg{event} = "Content scan ($report)";
	}

	security_log(\%msg, $recipients_aref);
}

1;
