--[[
Copyright (c) 2000-2017 Synology Inc. All rights reserved.
]]--

-- A plugin that implements basic cache
local settings = {
  action = 'reject',
  expire = 86400, -- 1 day by default
  key_prefix = 'cache',
  message = 'Spam result of message is in the cache',
  symbol = 'CACHE',
}

local rspamd_logger = require "rspamd_logger"
local rspamd_util = require "rspamd_util"
local hash = require "rspamd_cryptobox_hash"
local redis_params

local exports = {}
exports.settings = settings

local function make_key(task)
  return settings['key_prefix'] .. task:get_digest()
end

local function cache_check(task)
  local function redis_get_cb(err, data)
    if err ~= nil then
      rspamd_logger.errx(task, 'redis_get_cb received error: %1', err)
      return
    end
    if data ~= nil and type(data) == 'string' then
      -- Hash was found
      task:insert_result(settings['symbol'], 0.0, data)
      task:set_pre_result(settings['action'], settings['message'])
    end
  end

  local key = make_key(task)

  local ret = rspamd_redis_make_request(task,
    redis_params, -- connect params
    key, -- hash key
    false, -- is write
    redis_get_cb, -- callback
    'GET', -- command
    {key} -- arguments
  )

  if not ret then
    rspamd_logger.errx(task, "redis request wasn't scheduled")
  end
end

local function cache_set(task, spam_status)
  local function redis_set_cb(err)
    if err ~= nil then
      rspamd_logger.errx(task, 'redis_set_cb received error: %1', err)
    end
  end

  if not (spam_status and type(spam_status) == 'string') then
    rspamd_logger.errx(task, 'storing non-string data to redis: %1', spam_status)
    return
  end

  local key = make_key(task)
  rspamd_logger.infox(task, 'storing spam_status to cache')
  local ret = rspamd_redis_make_request(task,
    redis_params, -- connect params
    key, -- hash key
    true, -- is write
    redis_set_cb, -- callback
    'SETEX', -- command
    {key, tostring(settings['expire']), spam_status} -- arguments
  )
  if not ret then
    rspamd_logger.errx(task, "redis request wasn't scheduled")
  end
end
exports.set = cache_set

local opts = rspamd_config:get_all_opt('cache')
if not (opts and type(opts) == 'table') then
  rspamd_logger.infox(rspamd_config, 'module is unconfigured')
  exports.enabled = false
else
  redis_params = rspamd_parse_redis_server('cache')
  if not redis_params then
    rspamd_logger.infox(rspamd_config, 'no servers are specified, disabling module')
    exports.enabled = false
  else
    local id = rspamd_config:register_symbol({
      name = 'CACHE_CHECK',
      type = 'prefilter',
      callback = cache_check,
      priority = 1
    })
    exports.enabled = true
  end

  for k,v in pairs(opts) do
    settings[k] = v
  end
end

return exports
