#!/usr/bin/env python
# -*- encoding=utf8 -*-

from __future__ import print_function
import os, sys, csv

def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

def checkUserName(username):
    return -1 == username.find('/') and -1 == username.find(':') and -1 == username.find(',')

def unicode_csv_reader(unicode_csv_data, dialect=csv.excel, **kwargs):
    # csv.py doesn't do Unicode; encode temporarily as UTF-8:
    csv_reader = csv.reader(utf_8_encoder(unicode_csv_data),
                            dialect=dialect, **kwargs)
    for row in csv_reader:
        # decode UTF-8 back to Unicode, cell by cell:
        yield [unicode(cell, 'utf-8') for cell in row]

def utf_8_encoder(unicode_csv_data):
    for line in unicode_csv_data:
        yield line.encode('utf-8')

def main():
    import argparse, json, io, codecs

    parser = argparse.ArgumentParser(usage='%(prog)s [options] <csv file>')
    parser.add_argument('-n', '--name', nargs=1, help='generate the json format user list for the task')
    parser.add_argument('args', nargs=argparse.REMAINDER)
    args = parser.parse_args()

    #must specify the path of the csv file
    if 1 != len(args.args):
        parser.print_help()
        exit(-1)

    csv_file_path = args.args[0]
    if not os.path.isfile(csv_file_path):
        eprint('file not found [{0:s}]'.format(csv_file_path))
        exit(-1)

    is_check_only = True
    user_list_path = ''
    if None != args.name and len(args.name) > 0:
        is_check_only = False
        task_data_dir = os.path.realpath('/var/packages/MailPlus-Server/target/var/migration_data')
        task_dir = os.path.realpath(os.path.join(task_data_dir, args.name[0]))
        if task_data_dir != os.path.dirname(task_dir) or not os.path.isdir(task_dir):
            eprint('invalid task dir [{0:s}]'.format(task_dir))
            exit(-1)
        user_list_path = os.path.join(task_dir, 'user_list')

    user_info_list = list()
    src_acnt_set = set()
    dest_acnt_set = set()

    bytes = min(32, os.path.getsize(csv_file_path))
    raw = open(csv_file_path, 'rb').read(bytes)
    if raw.startswith(codecs.BOM_UTF8):
        encoding = 'utf-8-sig'
    else:
        encoding = 'utf8'

    with io.open(csv_file_path, 'r', encoding=encoding) as fp:
        for row in unicode_csv_reader(fp):
            try:
                src_acnt, src_passwd, dest_acnt = row

                #duplicate record is not allowed
                if src_acnt in src_acnt_set or dest_acnt in dest_acnt_set:
                    raise ValueError

                if not (checkUserName(src_acnt) and checkUserName(dest_acnt)):
                    continue

                src_acnt_set.add(src_acnt)
                dest_acnt_set.add(dest_acnt)

                if not is_check_only:
                    user_info = {'src_acnt': src_acnt.encode('utf8'), 'src_passwd': src_passwd.encode('utf8'), 'dest_acnt': dest_acnt.encode('utf8')}
                    user_info_list.append(user_info)
            except ValueError as e:
                eprint('parse error for "{0}"'.format(row))
                continue

    if not is_check_only:
        with open(user_list_path, 'w') as outfile:
            json.dump(user_info_list, outfile)
        os.chmod(user_list_path, 0o600)

if __name__ == '__main__':
    main()
