#!/usr/bin/env python
# -*- encoding=utf8 -*-
from __future__ import print_function
import os
import sys

smtp_config = {}
user_list = {}
task_config = {}

success_msg_format = u'''\
Dear user,

Mail migration from {0} to {1} succeeded.

Sincerely,
Synology DiskStation\
'''

failure_msg_format = u'''\
Dear user,

Mail migration from {0} to {1} failed.
For more information, please go to MailPlus Server.

Sincerely,
Synology DiskStation\
'''

def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

def processAccountName(account_name):
    if -1 != account_name.find('"'):
        return account_name

    at_index = account_name.find('@')
    backslash_index = account_name.find('\\')

    if -1 != at_index:
        result_name = account_name[:at_index]
    elif -1 != backslash_index and backslash_index < len(account_name):
        result_name = account_name[backslash_index + 1:]
    else:
        result_name = account_name
    return result_name

def sendMailNotify(user_item, notify_setting, is_sync_success):
    from email.MIMEText import MIMEText
    from email import Utils, Charset
    import subprocess

    recipient_list = []
    if notify_setting['is_to_src']:
        addr = u'{0:s}@{1:s}'.format(processAccountName(user_item['src_acnt']), task_config['server_setting']['addr'])
        recipient_list.append(addr)
    if notify_setting['is_to_dest']:
        addr = u'{0:s}@{1:s}'.format(processAccountName(user_item['dest_acnt']), smtp_config['smtp/smtp_main_domain'])
        recipient_list.append(addr)
    if notify_setting['is_to_addr']:
        addr = notify_setting['addr']
        recipient_list.append(addr)

    if 0 == len(recipient_list):
        return

    if is_sync_success:
        notify_message = success_msg_format.format(user_item['src_acnt'], user_item['dest_acnt'])
    else:
        notify_message = failure_msg_format.format(user_item['src_acnt'], user_item['dest_acnt'])

    sender = u'postmaster@{0:s}'.format(smtp_config['smtp/smtp_main_domain'])
    text_subtype = 'plain'
    try:
        Charset.add_charset('utf-8', Charset.QP, Charset.QP, 'utf-8')
        msg = MIMEText(notify_message.encode('utf8'), text_subtype, 'utf8')
        msg['Subject'] = u'Mail migration from {0} {1}'.format(user_item['src_acnt'], 'succeeded' if is_sync_success else 'failed')
        msg['From'] = sender
        msg['To'] = ', ' . join(recipient_list).encode('utf8')
        msg['Message-ID'] = Utils.make_msgid()

        args = ['/var/packages/MailPlus-Server/target/sbin/sendmail', '-t']
        sp = subprocess.Popen(args, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        sp.communicate(msg.as_string())
        sp.wait()
    except Exception, exc:
        eprint('mail failed; %s' % str(exc))

def sendDSMNotify(str_key, user_item):
    import shlex
    import subprocess

    cmd = '/var/packages/MailPlus-Server/target/bin/syno_send_notification -e "{0:s}" "%DEST_ACCOUNT%={1:s}" "%SOURCE_ACCOUNT%={2:s}"'.format(str_key, user_item['dest_acnt'], user_item['src_acnt'])
    try:
        cmd_list = shlex.split(cmd)
    except ValueError:
        eprint('failed to prepare cmd')

    try:
        ret_code = subprocess.call(cmd_list, shell=False)
        if 0 != ret_code:
            eprint('failed to send dsm notification')
    except:
        eprint('failed to send dsm notification')

def checkAndSendNotify(user_item, is_sync_success):

    if task_config['success_notify']['enable_notify'] and is_sync_success:
        notify_setting = task_config['success_notify']
        sendMailNotify(user_item, notify_setting, is_sync_success)
        if notify_setting['is_to_dsm']:
            sendDSMNotify('MAILPLUS_SERVER_MAIL_MIGRATION_SUCCEED', user_item)

    if task_config['error_notify']['enable_notify'] and not is_sync_success:
        notify_setting = task_config['error_notify']
        sendMailNotify(user_item, notify_setting, is_sync_success)
        if notify_setting['is_to_dsm']:
            sendDSMNotify('MAILPLUS_SERVER_MAIL_MIGRATION_FAIL', user_item)

def readJsonFile(task_name, json_file_path):
    import json

    json_data = None

    if not os.path.isfile(json_file_path):
        eprint('file [{0:s}] does not exist'.format(json_file_path))
        exit(-1)

    with open(json_file_path, 'r') as in_file:
        try:
            json_data = json.load(in_file)
        except ValueError:
            eprint('cannot parse file [{0:s}]'.format(json_file_path))
            exit(-1)

    return json_data

def main():
    import argparse
    import os

    global smtp_config, task_config, user_list
    parser = argparse.ArgumentParser(usage='%(prog)s [options]')
    parser.add_argument('-t', '--task', action='store', help='task name')
    parser.add_argument('-a', '--account', action='store', help='account name')
    parser.add_argument('-s', '--success', action='store_true', dest='is_sync_success', default=False, help='is task finishes successfully')
    args = parser.parse_args()

    if None == args.task or None == args.account:
        parser.print_help()
        exit(0)

    task_name = args.task
    account_name = args.account.decode('utf8')

    user_list_path = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, 'user_list')
    user_list = readJsonFile(task_name, user_list_path)
    task_config_path = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, 'config')
    task_config = readJsonFile(task_name, task_config_path)
    smtp_config_path = '/var/packages/MailPlus-Server/etc/mailconf.d/smtp'
    smtp_config = readJsonFile(task_name, smtp_config_path)

    for user_item in user_list:
        user_account_name = user_item['src_acnt'] + '->' + user_item['dest_acnt']
        if account_name == user_account_name:
            checkAndSendNotify(user_item, args.is_sync_success)
            break

if __name__ == '__main__':
    main()
