#################################################################################
#
# This script is modify from ExportCASConfig.ps1.
# This script is used to export config on the exchange server. The exported config
# will be import in Synology MailPlus-Server package to help you set up your
#  MailPlus-Server Environment.
#
#################################################################################


#
# Synopsis: Exports echange configuration to an XML file, and distribution group data
# to a format that MailPlus-Server recognizes.
#
# Usage:
#    .\ExchangeConfigExport.ps1
#

param([Boolean] $debug = $False)

#############################################################################
# Logs the information to the cloneLogFile.log
# isHost : If true Write the message to the host, inaddition to the log file.
# message : The message string to be logged.
#############################################################################
function Write-Information([Boolean] $isHost, [String] $message)
{
    # Log all the messages to the log file.
    if($logger -ne $null)
    {
        $logger.WriteLine()
        $logger.WriteLine($message)
    }

    # Write the error message to the host also.
    if($isHost -eq $true)
    {
        write-host $message
    }
}

###########################################################################
# The function adds some useful information into the log file
# before the actual Export process starts.
###########################################################################
function Write-LogStartInformation()
{
    $startTime = ([System.DateTime]::Now).ToString()
    Write-Information $false "************ BEGIN EXPORT PROCESS **************"
    Write-Information $false $startTime
}

#############################################################
# Releases the file handles used by the Export Script.
#############################################################
function Export-ReleaseHandles()
{
    write-debug "Export-ReleaseHandles"

	#Remove-PSSession Microsoft.Exchange.Management.PowerShell.E2010
	$existingSession = @(Get-PSSession | Where {$_.Availability -eq 'Available' -and $_.ConfigurationName -eq 'Microsoft.Exchange'})
	foreach ($session in $existingSession)
	{
		Remove-PSSession $session
	}

    if($logger -ne $null)
    {
        Write-Information $false "************ END EXPORT PROCESS **************"
        $logger.flush()
        $logger.close()
    }
    if($xmlWriter -ne $null)
    {
        $xmlWriter.Close()
    }

	if ($aliasWriter -ne $null)
	{
		$aliasWriter.close()
	}
}

###############################################################
# Finds the parameters of the objects the command returns. Gets
# the objects using the command, iterates through the parameters
# of each object and creates the XML component for each object.
# getCommand: The command to be run
# writer: Writer to write exported data to
###############################################################
function ExportObjects([String[]] $getCommand, [System.Xml.XmlTextWriter] $writer)
{
    write-debug ("ExportObjects " + $getCommand[0])
    # Get all the object we need to export using the given command.
    # The command may return:
    #   - null if there are no objects to export
    #   - single object
    #   - array of objects.

	$expression = $getCommand[0]

    $resultObjects = invoke-expression $expression
    # If no objects returned Log and return.
    if (!$resultObjects)
    {
        Write-Information $false "$getCommand returned 0 objects"
        return
    }


    # If we are here, there is at least one object to export.
    # First, extract the type of objects being exported.

	if ($getCommand[2] -eq "Array")
	{
		$writer.WriteStartElement($getCommand[1])
		$objectType = "Item"
	}
	else
	{
		$objectType = $getCommand[1]
	}

    foreach ($object in $resultObjects)
    {
        # If we have not done it yet, get all the object properties using the
        # Get-Member cmdlet.
        if ($objectPropertyNames -eq $null)
        {
			if ($getCommand[3] -ne $null)
			{
				$targetName = $getCommand[3].split()
				$objectPropertyNames = Get-Member -InputObject $object -MemberType Property -Name $targetName
			}
			else
			{
				$objectPropertyNames = Get-Member -InputObject $object -MemberType Property
			}
        }

        # Create a new XML element for the current object.
        $writer.WriteStartElement($objectType)

        # Save all the properties.
        foreach ($prop in $objectPropertyNames)
        {
            if ($object.($prop.Name) -eq $null)
            {
                $writer.WriteElementString($prop.Name, '$null')
            }
            elseif ($prop.Definition.ToString().ToUpper().Contains("MULTIVALUEDPROPERTY"))
            {
                if ($object.($prop.Name).Count -eq 0)
                {
                    $writer.WriteElementString($prop.Name, '$null')
                    continue
                }
                $multiValuedDataItem = $null
                foreach($multiValuedItem in $object.($prop.Name))
                {
                    if ($multiValuedDataItem -ne $null)
                    {
                        $multiValuedDataItem = $multiValuedDataItem + ", "
                    }
                    $multiValuedDataItem = $multiValuedDataItem +
                                           "'" +
                                           $multiValuedItem.ToString() +
                                           "'"
                }
                $writer.WriteElementString($prop.Name, $multiValuedDataItem)
            }
            else
            {
                $writer.WriteElementString($prop.Name, $object.($prop.Name))
            }
        }

        # Finish the element.
        $writer.WriteEndElement()

        # Increment the Objects Count counter.
        $resultObjectsCount++
    }
	if ($getCommand[2] -eq "Array")
	{
		$writer.WriteEndElement()
	}

    Write-Information $false "$getCommand returned $resultObjectsCount objects"

    return
}

###############################################################
# Get the distribution groups and their members, and create
# the XML component for each distribution groups.
# writer: Writer to write exported data to
###############################################################
function ExportDistributionGroup([System.IO.StreamWriter] $aliasWriter)
{
	write-debug ("ExportDistributionGroup")

	$distributionGroups = Get-DistributionGroup

	if (!$distributionGroups)
    {
        Write-Information $false "no distribution groups"
		$aliasWriter.flush()
        return
    }

	foreach ($distributionGroup in $distributionGroups)
	{
		if ($distributionGroup -eq $null)
		{
			continue
		}

		#composite alias member list
		$members = Get-DistributionGroupMember -identity $distributionGroup.Name
		$aliasMembers = $null
		foreach ($member in $members)
		{
			if ($member.RecipientType -eq 'UserMailbox')
			{
				$propName = 'SamAccountName'
			}
			elseif ($member.RecipientType -eq 'MailUniversalDistributionGroup' -or $member.RecipientType -eq 'MailUniversalSecurityGroup')
			{
				$propName = 'alias'
			}
			else
			{
				continue
			}

			if ($aliasMembers -eq $null) {
				$aliasMembers = $member.$propName
			}
			else
			{
				$aliasMembers = $aliasMembers + ' ,' + $member.$propName
			}
		}

		#write one alias mapping out
		if ($aliasMembers -ne $null)
		{
			$aliasWriter.writeLine($distributionGroup.Alias + ': ' + $aliasMembers)
		}
	}

	$aliasWriter.flush()

}

function writeEnabledElement([object] $targetObject, [System.Xml.XmlTextWriter] $writer)
{
	if ($targetObject.Enabled)
	{
		$writer.WriteElementString("Enabled", "True")
	}
	else
	{
		$writer.WriteElementString("Enabled", "False")
	}
}

function writeNotNullElement([object] $targetObject, [String] $propertyName, [System.Xml.XmlTextWriter] $writer)
{
	if ($targetObject -ne $null)
	{
		$writer.WriteElementString($propertyName, $targetObject)
	}
}

###############################################################
# Get the content filter config, and create the XML component
# writer: Writer to write exported data to
###############################################################
function ExportContentFilter([System.Xml.XmlTextWriter] $writer)
{
	write-debug ("ExportContentFilter")

	$writer.WriteStartElement("ContentFilterConfig")

	$ContentFilterConfig = Get-ContentFilterConfig
	writeEnabledElement $ContentFilterConfig $writer

	writeNotNullElement $ContentFilterConfig.BypassedRecipients "BypassedRecipients" $writer
	writeNotNullElement $ContentFilterConfig.BypassedSenders "BypassedSenders" $writer
	writeNotNullElement $ContentFilterConfig.BypassedSenderDomains "BypassedSenderDomains" $writer

	$phrases = Get-ContentFilterPhrase
	$writer.WriteStartElement("Phrases")
	foreach ($phrase in $phrases)
	{
		$writer.WriteStartElement("Item")
		$writer.WriteElementString("Influence", $phrase.Influence)
		$writer.WriteElementString("Phrase", $phrase.Phrase)
		$writer.WriteEndElement()
	}
	$writer.WriteEndElement()
	$writer.WriteEndElement()

}

###############################################################
# Get the ip block/allow list and ip block/allow list provider
# and create the XML component
# type: block/allow
# writer: Writer to write exported data to
###############################################################
function ExportIPListAndProvider([String] $type, [System.Xml.XmlTextWriter] $writer)
{
	#get settings
	if ($type -eq "block")
	{
		$ipListElement = "IPBlockList"
		$IPListConfig = Get-IPBlockListConfig
		$entries = Get-IPBlockListEntry
		$listProviderElement = "IPBlockListProvider"
		$IPListProvidersConfig = Get-IPBlockListProvidersConfig
		$providers = Get-IPBlockListProvider
	}
	else
	{
		$ipListElement = "IPAllowList"
		$IPListConfig = Get-IPAllowListConfig
		$entries = Get-IPAllowListEntry
		$listProviderElement = "IPAllowListProvider"
		$IPListProvidersConfig = Get-IPAllowListProvidersConfig
		$providers = Get-IPAllowListProvider
	}
	#write [block/allow] ip list
	$writer.WriteStartElement($ipListElement)
	writeEnabledElement $IPListConfig $writer
	foreach ($entry in $entries)
	{
		$writer.WriteStartElement("Entry")
		$writer.WriteElementString("IPRange", $entry.IPRange)
		$writer.WriteElementString("HasExpired", $entry.HasExpired)
		$writer.WriteEndElement()
	}
	$writer.WriteEndElement()

	#write [block/allow] ip list providers
	$writer.WriteStartElement($listProviderElement)
	writeEnabledElement $IPListProvidersConfig $writer
	foreach ($provider in $providers)
	{
		if ($provider.Enabled -eq $true)
		{
			$writer.WriteElementString("Provider", $provider.LookupDomain)
		}
	}
	$writer.WriteEndElement()

}

###############################################################
# Get the connection filter config, and create the XML component
# writer: Writer to write exported data to
###############################################################
function ExportConnectionFilter([System.Xml.XmlTextWriter] $writer)
{
	write-debug ("ExportConnectionFilter")


	$writer.WriteStartElement("ConnectionFilterConfig")

	ExportIPListAndProvider "block" $writer

	ExportIPListAndProvider "allow" $writer

	$writer.WriteEndElement()

}

###############################################################
# Get the sender filter config, and create the XML component
# writer: Writer to write exported data to
###############################################################
function ExportSenderFilter([System.Xml.XmlTextWriter] $writer)
{
	write-debug ("ExportSenderFilter")

	$SenderFilterConfig = Get-SenderFilterConfig

	if ($SenderFilterConfig -eq $null)
	{
		return
	}

	$writer.WriteStartElement("SenderFilterConfig")

	writeEnabledElement $SenderFilterConfig $writer

	if ($SenderFilterConfig -ne $null)
	{
		writeNotNullElement $SenderFilterConfig.BlockedSenders "BlockedSenders" $writer
		writeNotNullElement $SenderFilterConfig.BlockedDomains "BlockedDomains" $writer
		writeNotNullElement $SenderFilterConfig.BlockedDomainsAndSubdomains "BlockedDomainsAndSubdomains" $writer
	}

	$writer.WriteEndElement()

}

###############################################################
# Get the recipient filter config, and create the XML component
# writer: Writer to write exported data to
###############################################################
function ExportRecipientFilter([System.Xml.XmlTextWriter] $writer)
{
	write-debug ("ExportRecipientFilter")

	$RecipientFilterConfig = Get-RecipientFilterConfig

	if ($RecipientFilterConfig -eq $null)
	{
		return
	}

	$writer.WriteStartElement("RecipientFilterConfig")

	writeEnabledElement $RecipientFilterConfig $writer

	if ($RecipientFilterConfig -ne $null)
	{
		writeNotNullElement $RecipientFilterConfig.BlockListEnabled "BlockListEnabled" $writer
		writeNotNullElement $RecipientFilterConfig.BlockedRecipients "BlockedRecipients" $writer
	}

	$writer.WriteEndElement()

}

###############################################################
# Get the sender id filter config, and create the XML component
# writer: Writer to write exported data to
###############################################################
function ExportSenderIdFilter([System.Xml.XmlTextWriter] $writer)
{
	write-debug ("ExportSenderIdFilter")

	$SenderIdFilterConfig = Get-SenderIDConfig

	if ($SenderIdFilterConfig -eq $null)
	{
		return
	}

	$writer.WriteStartElement("SenderIdFilterConfig")

	writeEnabledElement $SenderIdFilterConfig $writer

	$writer.WriteEndElement()

}

################################################################
# Return the list of items to export, these items
# are associated with commands that get the config
# we care about directly.
# returns:  List of items to be exported.
################################################################
function directConfigItems()
{
    write-debug "directConfigItems"

	#each item should be a string array,
	#consist of command, result name used in xml, result type(Array or Object), optional fields to filter
	$exportItems = @(
	("Get-AcceptedDomain", "AcceptedDomain", "Array", "domainName DomainType default"),
	("Get-ReceiveConnector | Where {`$_.TransportRole -eq 'FrontendTransport'}", "SmtpSettings", "Array", "Name Bindings Enabled AuthMechanism PermissionGroups Fqdn MaxMessageSize MaxRecipientsPerMessage MaxHopCount MessageRateLimit MaxInboundConnectionPerSource"),
	("Get-ImapSettings", "ImapSettings", "Object", "UnencryptedOrTLSBindings SSLBindings"),
	("Get-PopSettings", "PopSettings", "Object", "UnencryptedOrTLSBindings SSLBindings"),
	("Get-Service | Where {`$_.Name -Like 'MSExchangeImap4*' -or `$_.Name -Like 'MSExchangePop3*' -or `$_.Name -Like 'MSExchangeFastSearch'}", "Services", "Array", "ServiceName Status"),
	("Get-TransportAgent", "AntiSpamAgents", "Array", "Identity Enabled"),
	("Get-TransportAgent -TransportService FrontEnd -identity 'Connection Filtering Agent'", "ConnectionFilterAgent", "Object", "Identity Enabled")
    )

    return $exportItems
}

################################################################
# Return the list of items to export, these items
# are associated with functions that compose the config we want
# in multiple steps.
# returns:  List of items to be exported.
################################################################
function indirectConfigItems()
{
    write-debug "indirectConfigItems"

	#each item should be a string array,
	#consist of command, result name used in xml, result type(Array or Object), optional fields to filter
	$exportItems = @(
	'ExportContentFilter',
	'ExportConnectionFilter',
	'ExportSenderFilter',
	'ExportRecipientFilter',
	'ExportSenderIdFilter'
    )

    return $exportItems
}


#############################################################################
# Import the cmdlet of exchange.
#############################################################################
function ImportExchangCmdlet()
{
	Add-PSSnapin Microsoft.Exchange.Management.PowerShell.E2010
	. $env:ExchangeInstallPath\bin\RemoteExchange.ps1
	Connect-ExchangeServer -auto
}


#############################################################################
# InitializationCheck
# Setup script variables, check arguments etc.
#############################################################################
function InitializationCheck()
{
    write-debug "InitializationCheck"
    # Global variable for storing handle to the clone log file.
    # Change this to Actual Path relative to [C:\Program Files\Microsoft\Exchange Server\]

    $script:success = $false

	$script:installPath = $env:exchangeinstallpath
    $script:logfilePath = $installPath + "Logging\SetupLogs\cloneLogFile.log"

    $script:logger = new-object System.IO.StreamWriter ($logfilePath , $true)

    Write-LogStartInformation

    $script:success = $true
}


#############################################################################
# Main Script starts here, validates the parameters. Defines the list of
# cloneable items. Gets the data corresponding
# to each item and adds them to the clone data file.
#############################################################################

#Usage
#:: ./ExchangeConfigExport.ps1
#

ImportExchangCmdlet

# Do all the Initialization Checks
InitializationCheck
if (-not $success)
{
    Export-ReleaseHandles
    exit
}

write-debug "Main"

# Create and initialize XML text writer.
$cloneConfigData = $pwd.path + "\SynologyExportedExchangeConf.xml"
$xmlWriter = new-object System.Xml.XmlTextWriter(
    $cloneConfigData,
    [System.Text.Encoding]::UTF8)

$xmlWriter.Formatting = [System.Xml.Formatting]::Indented
$xmlWriter.WriteStartDocument()
$xmlWriter.WriteStartElement("SynologExportedExchangeConfig")

$aliasPath = $pwd.path + "\SynologyExportedAlias.txt"
#truncate the file content
if (Test-Path $aliasPath)
{
	Clear-Content $aliasPath
}
$aliasWriter = new-object System.IO.StreamWriter ($aliasPath, $true)


# Read all get commands that we want to use for exporting configuration data
$getDirectCommands = directConfigItems
$getIndirectCommands = indirectConfigItems

$oldErrorActionPreference = $ErrorActionPreference
if (! $debug)
{
	$ErrorActionPreference = "SilentlyContinue"
}
# Export each category of objects.
foreach ($getCommand in $getDirectCommands)
{
	$commandName = $getCommand[0].split() | select -First 1
	if (Get-Command $commandName -errorAction SilentlyContinue)
	{
		ExportObjects $getCommand $xmlWriter
		# Exception handling.
		trap
		{
			$script:success = $false
			if ($debug)
			{
				Write-Information $true "Failed to execute '$getCommand'"
				Write-Information $true ("Reason: " + $error[0])
			}
		}
	}
}

# Export each category of objects.
foreach ($getCommand in $getIndirectCommands)
{
	$expression = $getCommand + " `$xmlWriter"
	invoke-expression $expression
	trap
	{
		$script:success = $false
		if ($debug)
		{
			Write-Information $true "Failed to execute '$getCommand'"
			Write-Information $true ("Reason: " + $error[0])
		}
	}
}

# Export alias data
ExportDistributionGroup $aliasWriter

$ErrorActionPreference = $oldErrorActionPreference

# Close the XML file.
$xmlWriter.WriteEndElement()
$xmlWriter.WriteEndDocument()
$xmlWriter.Close()

$aliasWriter.Close()

# Exception handling.
trap
{
    Write-Information $true "Exporting Exchange configuration information failed."
    Write-Information $true ("Reason: " + $error[0])
    Export-ReleaseHandles
    exit
}
if ($success)
{
	Write-Information $true "Exchange configuration is exported successfully to $cloneConfigData"
}
Export-ReleaseHandles
