#!/usr/bin/python
import sys
import os

sys.path.append("/var/packages/MailPlus-Server/target/backend_hook/")
from _Utils import *

#path define
MAILPLUS_SERVER_TARGET = "/var/packages/MailPlus-Server/target"

SDK_TOOL = MAILPLUS_SERVER_TARGET + "/bin/syno_personal_policy"
DOVECOT = MAILPLUS_SERVER_TARGET + "/scripts/daemon/dovecot.sh"
DOVECOT_AUTH_PATH = MAILPLUS_SERVER_TARGET + "/etc/dovecot/conf.d"
DOVECOT_POLICY_PATH = MAILPLUS_SERVER_TARGET + "/etc/dovecot/conf.d"

POSTFIX = MAILPLUS_SERVER_TARGET + "/scripts/daemon/postfix.sh"
POSTFIX_POLICY_PATH = MAILPLUS_SERVER_TARGET + "/etc"
POSTMAP = MAILPLUS_SERVER_TARGET + "/sbin/postmap"

# redis key
USER_POLICY_TREE = "user_policy_setting"

IMAP_DISABLE = "imap_enable_list-false"
IMAP_LOCAL = "imap_local_enable_list-true"
POP3_DISABLE = "pop3_enable_list-false"
POP3_LOCAL = "pop3_local_enable_list-true"

AUTO_FORWARD = "smtp_auto_forward_disable_list-false"
SMTP_LOCAL = "smtp_local_enable_list-true"

ATTACHMENT_SIZE_PREFIX = "attachment_size_list"
DAILY_QUOTA_PREFIX = "daily_quota_list"
DAILY_FLOW_PREFIX = "daily_flow_limit_list"

CANONICAL_DOMAIN = "canonical_domain"
MAIN_DOMAIN = "smtp_main_domain"

ACCOUNT_TYPE = "smtp/account_type"
ACCOUNT_DOMAIN = "smtp/acc_domain_name"
ACCOUNT_WIN_SHORT = "smtp/win_domain_short_name"
SMTP_DOMAIN = "smtp/smtp_main_domain"
SMTP_SELF_DOMAIN = "smtp/canonical_domain"
SMTP_HOST = "smtp/smtp_hostname"

def callBinaryService(service, action):
    cmd = "%s %s" % (service, action)
    result, _ = executeCommand(cmd, False)

class InfoProcess(object):
    def __init__(self):
        self.redisCall = BackendAPI()

    def _setFilePath(self, service, fileSubPath):
        if service == "postfix":
            return os.path.join(POSTFIX_POLICY_PATH, fileSubPath)
        elif service == "dovecot":
            return os.path.join(DOVECOT_POLICY_PATH, fileSubPath)
        elif service == "dovecot_auth":
            return os.path.join(DOVECOT_AUTH_PATH, fileSubPath)
        else:
            SYSLOG(LOG_ERR, "Unknown service type, please add the relative rule")
            exit(1)

    def _countBits(self, mask):
        transform = int(mask)
        bits = 8
        count = 0
        while bits > 0:
            bits = bits - 1
            if transform & 1:
                count = count + 1
            transform = transform >> 1
        return count

    def getLocalSection(self):
        import re
        bindIF = self.redisCall.getHostIF()
        hostIP = self.redisCall.getHostIP()
        ipConfig, _ = executeCommand("synonet --show", True)
        hostMask = re.search(bindIF + r".*?^Mask: (\d+)\.(\d+)\.(\d+)\.(\d+)", ipConfig, re.DOTALL | re.MULTILINE)
        section = 0
        if hostMask:
            section = self._countBits(hostMask.group(1)) + self._countBits(hostMask.group(2)) + self._countBits(hostMask.group(3)) + self._countBits(hostMask.group(4))
        return hostIP + "/" + str(section)

    def turnUidStringToFullName(self, uidString):
        from subprocess import Popen, STDOUT, PIPE
        p = Popen([SDK_TOOL, "getFullNameList"], stdout=PIPE, stdin=PIPE)
        userFullName = p.communicate(input=uidString)[0]
        if p.returncode:
            SYSLOG(LOG_ERR, "error with getting user full name list")
        if userFullName:
            return userFullName
        return ""

    def parseRedisKey(self, tree, prefixKey):
        import re
        listAllKey = self.redisCall.getTree(tree)
        if not listAllKey:
            return []
        prefixKeySet = re.findall(prefixKey + "[^\']*", str(listAllKey))
        return prefixKeySet

    def getRedisAsList(self, tree, key):
        userListString = self.redisCall.mailconfGet(tree + "/" + key)
        if userListString:
            uidList = self.turnUidStringToFullName(userListString)
            if uidList:
                return uidList.split(",")
        return []

    def getRedisPrefixKey(self, tree, prefixKey):
        keyListDiction = {}
        prefixKeySet = self.parseRedisKey(tree, prefixKey)
        if prefixKeySet:
            for key in prefixKeySet:
                value = self.redisCall.mailconfGet(tree + "/" + key)
                if value:
                    uidList = self.turnUidStringToFullName(value)
                    if uidList:
                        keyListDiction[key] = uidList.split(",")
        return keyListDiction

    def assignListValueToDict(self, userList, constValue):
        if userList:
            return {diction:constValue for diction in userList}
        else:
            return {}

    def getAccountType(self):
        return self.redisCall.mailconfGet(ACCOUNT_TYPE)

    def getAccountDomain(self):
        return self.redisCall.mailconfGet(ACCOUNT_DOMAIN)

    def getAccountWinShort(self):
        return self.redisCall.mailconfGet(ACCOUNT_WIN_SHORT)

    def getHostName(self):
        return self.redisCall.mailconfGet(SMTP_HOST)

    def getDomain(self):
        return self.redisCall.mailconfGet(SMTP_DOMAIN)

    def getSelfDefineDomain(self):
        return self.redisCall.mailconfGet(SMTP_SELF_DOMAIN)

    def getFQDN(self):
        accType = self.getAccountType()
        if accType == "ldap":
            return "@" + self.getAccountDomain()
        elif accType == "win":
            return self.getAccountWinShort() + "\\"
        elif accType == "local":
            return ""
        else:
            SYSLOG(LOG_ERR, "Unknown account type, please check redis account type key")
            exit(1)

    def handleDoubleEscape(self, fullNameList):
        import re
        fixList = []
        regex = re.compile("\\\\")
        if fullNameList:
            for fullName in fullNameList:
                fixList.append(regex.sub("\\\\\\\\", fullName))
        return fixList

    def turnFullNameToAllDomainName(self, fullNameList):
        import re
        fqdn = self.getFQDN()
        canonicalDomain = self.getSelfDefineDomain()
        allDomain = canonicalDomain.split(", ") if canonicalDomain else []
        allDomain.append(self.getDomain())
        accType = self.getAccountType()
        if accType == "win":
            fqdn = fqdn + "\\"
        regex = re.compile(fqdn)
        retList = []
        if fullNameList and allDomain:
            if accType == "win":
                for fullName in fullNameList:
                    for domain in allDomain:
                        retList.append(regex.sub("", fullName) + "@" + domain)
            elif accType == "ldap":
                for fullName in fullNameList:
                    for domain in allDomain:
                        retList.append(regex.sub("@" + domain, fullName))
            else:
                for fullName in fullNameList:
                    for domain in allDomain:
                        retList.append(fullName + "@" + domain)
        return retList

    def extractSufixNum(self, prefixDict, magnification):
        import re
        retDict = {}
        if prefixDict:
            for policy, userList in prefixDict.items():
                result = re.search(r"-(-*\d+)", policy)
                if result:
                    limitNum = long(result.group(1)) * magnification
                else:
                    continue
                if userList:
                    retDict.update(self.assignListValueToDict(userList, str(limitNum)))
        return retDict

    def setUserFullName(self, userList, fqdnString):
        if not userList:
            return []
        if self.getAccountType() == "win":
            return [fqdnString + user for user in userList]
        else:
            return [user + fqdnString for user in userList]

    def dumpKeyValAsFile(self, path, keyValDict):
        import fcntl
        try:
            with open(path, 'w') as genFile:
                fcntl.flock(genFile, fcntl.LOCK_EX | fcntl.LOCK_NB)
                if not keyValDict:
                    return
                for key, value in keyValDict.items():
                    if key == "":
                        continue
                    genFile.write(str(key) + " " + str(value) + "\n")
        except Exception as e:
            SYSLOG(LOG_ERR, "gen user diction restriciton file error (" + str(e) + ")")

    def dumpListAsFile(self, path, listVal):
        import fcntl
        try:
            with open(path, 'w') as genFile:
                fcntl.flock(genFile, fcntl.LOCK_EX | fcntl.LOCK_NB)
                if not listVal:
                    return
                for value in listVal:
                    if value == "":
                        continue
                    genFile.write(str(value) + "\n")
        except Exception as e:
            SYSLOG(LOG_ERR, "gen user list restriction file error (" + str(e) + ")")

    def checkDovecotSwitch(self):
        imap_disable = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + IMAP_DISABLE)
        pop3_disable = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + POP3_DISABLE)
        imap_local = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + IMAP_LOCAL)
        pop3_local = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + POP3_LOCAL)

        if (imap_disable and imap_disable != "") or (pop3_disable and pop3_disable != ""):
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_deny", "yes")
        else:
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_deny", "no")

        if (imap_local and imap_local != "") or (pop3_local and pop3_local != ""):
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_local", "yes")
        else:
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_local", "no")

    def checkPostfixSwitch(self):
        send_local = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + SMTP_LOCAL);
        if send_local and send_local != "":
            self.redisCall.mailconfSet("smtp/smtpd_send_local_only", "yes")
        else:
            self.redisCall.mailconfSet("smtp/smtpd_send_local_only", "no")


class PostfixPolicy(object):
    def __init__(self):
        self.change = False
        self.infoUtil = InfoProcess()
        self.domain = "@" + self.infoUtil.getDomain()

    def regenAllFile(self):
        self.genAutoForwardConf()
        self.genSendLocalOnlyConf()
        self.genSenderQuotaConf()
        self.genFlowLimitConf()
        self.genAttachmentLimitConf()

    def setUserPolicy(self):
        if (isKeyChange(AUTO_FORWARD) or isKeyChange(MAIN_DOMAIN) or isKeyChange(CANONICAL_DOMAIN)):
            self.genAutoForwardConf()

        if (isKeyChange(SMTP_LOCAL) or isKeyChange(MAIN_DOMAIN) or isKeyChange(CANONICAL_DOMAIN)):
            self.genSendLocalOnlyConf()
            self.change = True

        if (isPrefixKeyChange(DAILY_QUOTA_PREFIX)):
            self.genSenderQuotaConf()

        if (isPrefixKeyChange(DAILY_FLOW_PREFIX)):
            self.genFlowLimitConf()

        if (isPrefixKeyChange(ATTACHMENT_SIZE_PREFIX)):
            self.genAttachmentLimitConf()

        if (isTreeChange(USER_POLICY_TREE)):
            callBinaryService(POSTFIX, "reload")

        if (self.change):
            self.infoUtil.checkPostfixSwitch()

    def genAutoForwardConf(self):
        filePath = self.infoUtil._setFilePath("postfix", "user_forward")
        shortList = self.infoUtil.getRedisAsList(USER_POLICY_TREE, AUTO_FORWARD)
        userKeyVal = {}
        if shortList:
            shortList = self.infoUtil.turnFullNameToAllDomainName(shortList)
            userKeyVal = self.infoUtil.assignListValueToDict(shortList, "enable")
        self.infoUtil.dumpKeyValAsFile(filePath, userKeyVal)
        callBinaryService(POSTMAP, filePath)

    def genSendLocalOnlyConf(self):
        # gen user restriction file
        userFilePath = self.infoUtil._setFilePath("postfix", "user_local_only")
        shortList = self.infoUtil.getRedisAsList(USER_POLICY_TREE, SMTP_LOCAL)
        userKeyVal = {}
        if shortList:
            userKeyVal = self.infoUtil.assignListValueToDict(shortList, "local_only")
        self.infoUtil.dumpKeyValAsFile(userFilePath, userKeyVal)
        callBinaryService(POSTMAP, userFilePath)

        # gen permission domain file
        domainFilePath = self.infoUtil._setFilePath("postfix", "permit_domain")
        myDomain = self.infoUtil.getDomain()
        allDomain = {myDomain : "OK", "localhost." + myDomain : "OK"}
        allDomain.update(self.infoUtil.assignListValueToDict(self.infoUtil.getSelfDefineDomain().split(", "), "OK"))
        self.infoUtil.dumpKeyValAsFile(domainFilePath, allDomain)
        callBinaryService(POSTMAP, domainFilePath)

    def genSenderQuotaConf(self):
        filePath = self.infoUtil._setFilePath("postfix", "sender_quota_map")
        shortDict = self.infoUtil.getRedisPrefixKey(USER_POLICY_TREE, DAILY_QUOTA_PREFIX)
        extendDict = {}
        if shortDict:
            extendDict = self.infoUtil.extractSufixNum(shortDict, 1)
        self.infoUtil.dumpKeyValAsFile(filePath, extendDict)
        callBinaryService(POSTMAP, filePath)

    def genFlowLimitConf(self):
        filePath = self.infoUtil._setFilePath("postfix", "flow_limit")
        shortDict = self.infoUtil.getRedisPrefixKey(USER_POLICY_TREE, DAILY_FLOW_PREFIX)
        extendDict = {}
        if shortDict:
            # MB turns into Byte
            extendDict = self.infoUtil.extractSufixNum(shortDict, 1048576)
        self.infoUtil.dumpKeyValAsFile(filePath, extendDict)
        callBinaryService(POSTMAP, filePath)

    def genAttachmentLimitConf(self):
        filePath = self.infoUtil._setFilePath("postfix", "attachment_limit")
        shortDict = self.infoUtil.getRedisPrefixKey(USER_POLICY_TREE, ATTACHMENT_SIZE_PREFIX)
        extendDict = {}
        if shortDict:
            # MB turns into Byte and multiply increase of base64 encode -> *4/3
            extendDict = self.infoUtil.extractSufixNum(shortDict, 1398101)
        self.infoUtil.dumpKeyValAsFile(filePath, extendDict)
        callBinaryService(POSTMAP, filePath)
        
class DovecotPolicy(object):
    def __init__(self):
        self.change = False
        self.infoUtil = InfoProcess()
        self.fqdn = self.infoUtil.getFQDN()

    def setUserPolicy(self):
        if (isKeyChange(IMAP_DISABLE) or isKeyChange(POP3_DISABLE)):
            self.genLimitedUserConf(IMAP_DISABLE, "disable_imap")
            self.genLimitedUserConf(POP3_DISABLE, "disable_pop3")
            self.change = True
        if (isKeyChange(IMAP_LOCAL) or isKeyChange(POP3_LOCAL)):
            self.genLimitedUserConf(IMAP_LOCAL, "local_imap")
            self.genLimitedUserConf(POP3_LOCAL, "local_pop3")
            self.genLocalPassDbConf()
            self.change = True
        self.infoUtil.checkDovecotSwitch()
        return self.change

    def genLocalPassDbConf(self):
        import fcntl
        filePath = self.infoUtil._setFilePath("dovecot_auth", "auth-local.conf.ext")
        try:
            with open(filePath, 'w') as confFile:
                fcntl.flock(confFile, fcntl.LOCK_EX | fcntl.LOCK_NB)
                confFile.write("passdb {\n\
                        driver = passwd-file\n\
                        args = %s \n\
                        default_fields = nopassword\n\
                        result_success = continue-ok\n\
                        result_failure = continue-fail\n\
                }\n" % (DOVECOT_POLICY_PATH + "/local_%s"))

                confFile.write("passdb {\n\
                        driver = static\n\
                        default_fields = nopassword allow_nets=127.0.0.1,%s\n\
                        skip = unauthenticated\n\
                        result_success = continue-fail\n\
                        result_failure = return-fail\n\
                }" % (self.infoUtil.getLocalSection()))

        except Exception as e:
            SYSLOG(LOG_ERR, "gen dovecot auth-local fail (" + str(e) + ")")

    def genLimitedUserConf(self, key, fileName):
        filePath = self.infoUtil._setFilePath("dovecot", fileName)
        shortList = self.infoUtil.getRedisAsList(USER_POLICY_TREE, key)
        shortList = self.infoUtil.handleDoubleEscape(shortList)
        self.infoUtil.dumpListAsFile(filePath, shortList)

def usage():
    print 'If there is no parameter, it runs when redis key change'
    print 'Or use [regenPostfixPolicy]'

if __name__ == '__main__':
    postfixPart = PostfixPolicy()
    dovecotPart = DovecotPolicy()
    if len(sys.argv) < 2:
        postfixPart.setUserPolicy()

        if (dovecotPart.setUserPolicy()):
            os.system("/var/packages/MailPlus-Server/target/scripts/daemon/dovecot.sh restart")
    elif len(sys.argv) == 2 and sys.argv[1] == 'regenPostfixPolicy':
        postfixPart.regenAllFile()
    else:
        usage()
