#!/bin/sh

if [ -d "$1" ]; then
	echo "change directory to $1"
	cd "$1"
fi

openssl=`which openssl`
days="7200"
certversion="3"

CA_CRT="ca.crt"
CA_KEY="ca.key"
SVR_CRT="server.crt"
SVR_KEY="server.key"
SVR_CSR="server.csr"
SYS_CA_CRT="/usr/syno/etc/ssl/ssl.crt/$CA_CRT"
SYS_CA_KEY="/usr/syno/etc/ssl/ssl.key/$CA_KEY"

# $1: config file name
gen_openssl_config()
{
	config_file=$1
	cat > $config_file <<EOT
[ req ]
default_bits                    = 1024
distinguished_name              = req_DN
[ req_DN ]
countryName                     = "1. Country Name             (2 letter code)"
countryName_default             = XY
countryName_min                 = 2
countryName_max                 = 2
stateOrProvinceName             = "2. State or Province Name   (full name)    "
stateOrProvinceName_default     = Snake Desert
localityName                    = "3. Locality Name            (eg, city)     "
localityName_default            = Snake Town
0.organizationName              = "4. Organization Name        (eg, company)  "
0.organizationName_default      = Snake Oil, Ltd
organizationalUnitName          = "5. Organizational Unit Name (eg, section)  "
organizationalUnitName_default  = Certificate Authority
commonName                      = "6. Common Name              (eg, CA name)  "
commonName_max                  = 64
commonName_default              = Snake Oil CA
emailAddress                    = "7. Email Address            (eg, name@FQDN)"
emailAddress_max                = 40
emailAddress_default            = ca@snakeoil.dom
[ x509v3 ]
subjectAltName   = email:copy
basicConstraints = CA:true,pathlen:0
nsComment        = "mod_ssl generated custom CA certificate"
nsCertType       = sslCA
EOT
}

gen_ext_config()
{
	config_file=$1
	cat > $config_file << EOT
extensions = x509v3_ssl_server
[ x509v3_ssl_server ]
subjectAltName   = email:copy
nsComment        = "mod_ssl generated custom server certificate"
nsCertType       = server
EOT
}

SSL_CONFIG="openssl.conf"
EXT_CONFIG="ext.conf"

echo "Prepare openssl config file: $SSL_CONFIG"
	gen_openssl_config $SSL_CONFIG
	gen_ext_config $EXT_CONFIG
echo "______________________________________________________________________"
echo "STEP 1: Generating X.509 certificate/key for CA [ca.crt/ca.key]"
	if [ -e $SYS_CA_CRT -a -e $SYS_CA_KEY ]; then
		echo "copy CA certificate/key from ssl..."
		cp $SYS_CA_CRT $CA_CRT
		cp $SYS_CA_KEY $CA_KEY
	else
		if [ ".$certversion" = .3 -o ".$certversion" = . ]; then
			ext_section="-extensions x509v3"
		fi
		$openssl req -new -x509 -sha1 \
			-batch \
			-config $SSL_CONFIG \
			-days $days \
			-newkey rsa:1024 -rand /dev/urandom -nodes \
			-keyout $CA_KEY \
			$ext_section \
			-out $CA_CRT
		if [ $? -ne 0 ]; then
			echo "mkcert.sh:Error: Failed to generate certificate signing request" 1>&2
			exit 1
		fi
	fi

	echo "Verify: matching certificate & key modulus"
	modcrt=`$openssl x509 -noout -modulus -in $CA_CRT | sed -e 's;.*Modulus=;;'`
	modkey=`$openssl rsa -noout -modulus -in $CA_KEY | sed -e 's;.*Modulus=;;'`
	if [ ".$modcrt" != ".$modkey" ]; then
		echo "mkcert.sh:Error: Failed to verify modulus on resulting X.509 certificate" 1>&2
		exit 1
	fi

	echo "Verify: matching certificate signature"
	$openssl verify $CA_CRT
	if [ $? -ne 0 ]; then
		echo "mkcert.sh:Error: Failed to verify signature on resulting X.509 certificate" 1>&2
		exit 1
	fi
echo "______________________________________________________________________"
echo "STEP 2: Generating private key and certificate signing request for SERVER (1024 bit) [server.key]"
	$openssl req -new -sha1 \
		-config $SSL_CONFIG \
		-batch \
		-newkey rsa:1024 -rand /dev/urandom -nodes \
		-keyout $SVR_KEY \
		-out $SVR_CSR
	if [ $? -ne 0 ]; then
		echo "mkcert.sh:Error: Failed to generate certificate signing request" 1>&2
		exit 1
	fi
echo "______________________________________________________________________"
echo "STEP 3: Generating X.509 certificate signed by own CA [server.crt]"
	extfile=""
	if [ ".$certversion" = .3 -o ".$certversion" = . ]; then
		ext_section="-extfile $EXT_CONFIG"
	fi
	$openssl x509 -req \
		-days $days \
		$ext_section \
		-out $SVR_CRT \
		-in $SVR_CSR \
		-CA $CA_CRT \
		-CAkey $CA_KEY \
		-CAcreateserial
	if [ $? -ne 0 ]; then
		echo "mkcert.sh:Error: Failed to generate X.509 certificate" 1>&2
		exit 1
	fi

	echo "Verify: matching certificate & key modulus"
	modcrt=`$openssl x509 -noout -modulus -in $SVR_CRT | sed -e 's;.*Modulus=;;'`
	modkey=`$openssl rsa -noout -modulus -in $SVR_KEY | sed -e 's;.*Modulus=;;'`
	if [ ".$modcrt" != ".$modkey" ]; then
		echo "mkcert.sh:Error: Failed to verify modulus on resulting X.509 certificate" 1>&2
		exit 1
	fi

	echo "Verify: matching certificate signature"
	$openssl verify -CAfile $CA_CRT $SVR_CRT
	if [ $? -ne 0 ]; then
		echo "mkcert.sh:Error: Failed to verify signature on resulting X.509 certificate" 1>&2
		exit 1
	fi
echo "______________________________________________________________________"
echo ""

rm -rf $SSL_CONFIG
rm -rf $EXT_CONFIG
