#!/bin/sh
#
# Usage:
#        handler_sb.sh safemode [ --boot ]
#			After 5.0 hotfix 1, SHA support safemode to handle split-brain
#
#        handler_sb.sh unbind
#
# return 1: error, unbind local
#        2: standby failed
#
# vim:ft=sh


HA_PREFIX=/usr/syno/synoha

. $HA_PREFIX/etc.defaults/rc.subr
PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/syno/sbin:/usr/syno/bin:/usr/local/sbin:/usr/local/bin:/usr/syno/synoha/sbin

SYNO_HA_SAFEMODE=$HA_SAFEMODE_INFO
SYNO_HA_PREV_ACTIVE=$PREVIOUS_ROLE_ACTIVE
SYNO_HA_PREV_PASSIVE=$PREVIOUS_ROLE_PASSIVE
SYNO_HA_ORIGINAL_HOSTNAME_FILE=$CRM_HOSTNAME_ORIGINAL
SB_FLOCK='/tmp/.ha.handler_sb.lock'

lock()
{
	if [ -f "$SB_FLOCK" ]; then
		local prev_lock=`cat $SB_FLOCK`
		synoha_log "SB: handler_sb.sh lock ($1) failed, pervious lock pid:$prev_lock"
		return $LSB_ERR_GENERIC
	fi

	trap '/bin/rm "$SB_FLOCK" &> /dev/null' INT TERM EXIT ABRT
	echo "$$ $1" > "$SB_FLOCK"
	return $LSB_SUCCESS
}

unlock()
{
	/bin/rm "$SB_FLOCK" &> /dev/null
}

Unbind()
{
	synoha_log "SB: Split brain handler failed: $1"
	$SYNOHA_BIN --handle-sb &
	exit 1
}

ClearHAStates()
{
	synoha_log "SB: Clear HA status"
	crm resource cleanup DRBD          &> /dev/null
	crm -F node clearstate $LOCAL_HOST &> /dev/null
}

send_cib_info()
{
	while :; do
		[ -f "$SYNO_HA_SAFEMODE" ] || exit 0

		if ! ps | grep "synoha --cib-nodeinfo-set" | grep -v grep &> /dev/null ; then
			$SYNOHA_BIN --cib-nodeinfo-set
		fi
		sleep 30
	done
}

start_handle_sb()
{
	local IsBOOT=$1

	# prevent multiple handle_sb
	if [ "--boot" != "$IsBOOT" ]; then
		[ -f "$SYNO_HA_SAFEMODE" ] && exit 0
	fi

	{
		synoha_log "SB: Split brain handler"

		[ -n "$NODE_HOST0" ] || Unbind "host0 missing"
		[ -n "$NODE_HOST1" ] || Unbind "host1 missing"

		mkdir -p `dirname $SYNO_HA_SAFEMODE`
		touch $SYNO_HA_SAFEMODE

		rm -f "$SYNO_HA_PREV_ACTIVE"
		rm -f "$SYNO_HA_PREV_PASSIVE"

		[ "--boot" != "$IsBOOT" ] && synoha --remote-run-safemode

		local constraint_t="<constraints><rsc_location id='RULE_SB_%s' node='%s' rsc='DUMMY_START' score='-INFINITY'/></constraints>"
		local standby_t="<instance_attributes id='nodes-%s'><nvpair id='nodes-%s-standby' name='standby' value='on'/></instance_attributes>"
		for h in "$NODE_HOST0" "$NODE_HOST1"; do
			# when bootup, only set constraint & standby to LOCAL_HOST
			[ "--boot" == "$IsBOOT" -a "$LOCAL_HOST" != "$h" ] && continue

			local cons=`printf "$constraint_t" $h $h`
			cibadmin -Mc --xml-text "$cons"

			local stby=`printf "$standby_t" $h $h`
			cibadmin -Mc --xml-text "$stby"
		done

		synoha_log "SB: Split brain handler done"


		_standby()
		{
			local _crm_mon="crm_mon -1"
			# Sync as `SZ_CRM_MON_NODE_STANDBY` in crm.cc
			local CRM_MON_NODE_STANDBY="$_crm_mon | grep -E \"standby|Online\" -A40 | grep -v standby | grep -v \".*node.*status.*\" | grep -v unmanaged | synohagrep %s"

			local _localhost_stby=`printf "$CRM_MON_NODE_STANDBY" $LOCAL_HOST`
			local _host0_stby=`printf "$CRM_MON_NODE_STANDBY" $NODE_HOST0`
			local _host1_stby=`printf "$CRM_MON_NODE_STANDBY" $NODE_HOST1`

			if [ "--boot" == "$IsBOOT" ]; then
				# only wait for local standby after reboot
				[ -z "`eval $_localhost_stby`" ]
			else
				[ -z "`eval $_host0_stby`" -a -z "`eval $_host1_stby`" ]
			fi
		}

		local _sleep_sec=5
		if ! HASleepFor _standby $MAX_HA_SERV_STOP_SEC $_sleep_sec &>/dev/null ; then
				synoha_log "SB: Standby failed, force reboot"
				crm_mon -n | synoha_log
				ClearHAStates
				/sbin/reboot -f
				exit 2
		fi

		ClearHAStates

		/sbin/start httpd-sys
		/usr/syno/bin/synobootseq --set-boot-done
		send_cib_info &

		synoha_log "SB: enter SHA safemode"
	}&

} # end of start_handle_sb()


lock $1 || exit $?

[ -e $HA_PREFIX/etc/ha.conf ] || Unbind "ha.conf missing"

action=$1; shift
case "$action" in
	safemode)
		start_handle_sb "$@"
		;;
	unbind)
		Unbind
		;;
	*)
		exit $LSB_ERR_ARGS
esac

exit $?
